DROP TABLE IF EXISTS scheduling.resource CASCADE; --wegen evtl bereits vorhandenen DBUpdates und Kompatibliltät
CREATE TABLE scheduling.resource (
  id          serial PRIMARY KEY,
  context     text NOT null,
  context_id  int NOT null,
  ta_fk       numeric(16,8) DEFAULT 1
);

CREATE UNIQUE INDEX resource__context_id            ON scheduling.resource( context, context_id ) /*INCLUDE (id)*/ ;
CREATE UNIQUE INDEX resource__context_id__where_x   ON scheduling.resource( context, context_id ) /*INCLUDE (id)*/ WHERE context IN ('ksvba');

-- Migrationsskript Terminierung: Spalten "a2w_resource_id_main_terminated" und "a2w_resource_id_main_fix" der Tabelle "ab2_wkstplan" anlegen.
-- Muss hier erfolgen, da Tabelle "scheduling.resource" schon da (Referenz) und noch keine Funktionen angelegt, welche die beiden Splaten benötigen.
DO $$
BEGIN
    IF NOT EXISTS ( SELECT true FROM information_schema.columns WHERE table_schema = 'public' AND table_name = 'ab2_wkstplan' AND column_name = 'a2w_resource_id_main_terminated' ) THEN
        ALTER TABLE ab2_wkstplan ADD COLUMN a2w_resource_id_main_terminated integer REFERENCES scheduling.resource ON DELETE SET null; -- (resource.id) resource.context_id = ksb_id
    END IF;
    IF NOT EXISTS ( SELECT true FROM information_schema.columns WHERE table_schema = 'public' AND table_name = 'ab2_wkstplan' AND column_name = 'a2w_resource_id_main_fix' ) THEN
        ALTER TABLE ab2_wkstplan ADD COLUMN a2w_resource_id_main_fix integer REFERENCES scheduling.resource ON DELETE SET null;
    END IF;
END $$ LANGUAGE plpgsql;

CREATE TABLE scheduling.resource_requirement (
  id                   serial PRIMARY KEY,

  required_by          int NOT NULL REFERENCES scheduling.resource ON UPDATE CASCADE ON DELETE CASCADE,
  amount               int DEFAULT 1,
  context              text NOT NULL
);

CREATE INDEX resource_requirement__context ON scheduling.resource_requirement ( context );
CREATE INDEX resource_requirement__required_by ON scheduling.resource_requirement ( required_by );

CREATE TABLE scheduling.resource_requirement_option (
  id              serial PRIMARY KEY,
  requirement_id  int NOT NULL REFERENCES scheduling.resource_requirement ON UPDATE CASCADE ON DELETE CASCADE,
  resource_id     int NOT NULL REFERENCES scheduling.resource ON UPDATE CASCADE ON DELETE CASCADE,
  load            numeric(12,6) DEFAULT 1
);

CREATE INDEX resource_requirement_option__requirement_id ON scheduling.resource_requirement_option ( requirement_id );
CREATE INDEX resource_requirement_option__resource_id ON scheduling.resource_requirement_option ( resource_id );

CREATE TABLE scheduling.vertex (
  id          serial PRIMARY KEY,
  ab2_a2_id   int REFERENCES ab2( a2_id ) ON DELETE CASCADE,
  resource_id int REFERENCES scheduling.resource( id ) ON DELETE CASCADE, -- Ressourcen-ID der Arbeitsplatz-Ressource
  load        numeric NOT NULL
);

CREATE INDEX vertex_ab2_a2_id_resource_id_idx ON scheduling.vertex( ab2_a2_id, resource_id );


CREATE TABLE scheduling.edge (
  id            serial PRIMARY KEY,
  vertex_from   int REFERENCES scheduling.vertex( id ) ON DELETE CASCADE,
  vertex_to     int REFERENCES scheduling.vertex( id ) ON DELETE CASCADE,
  length        numeric
);

CREATE INDEX edge_vertex_to_vertex_from_idx ON scheduling.edge ( vertex_to, vertex_from );
CREATE INDEX edge_vertex_from_vertex_to_idx ON scheduling.edge ( vertex_from, vertex_to );


CREATE TABLE scheduling.resource_timeline_scenario (
    id    serial PRIMARY KEY NOT NULL,
    name  text NOT NULL
);


CREATE TABLE scheduling.resource_timeline (
  ti_id           serial PRIMARY KEY,
  ti_a2_id        integer REFERENCES ab2( a2_id ) ON DELETE CASCADE,
  ti_resource_id  int REFERENCES scheduling.resource( id ),
  ti_date_start   timestamp NOT NULL,
  ti_date_end     timestamp NOT NULL,

  ti_type         scheduling.resource_timeline_blocktype DEFAULT 'task' NOT NULL,
  ti_ta_kf        numeric(16,6) DEFAULT 0,
  ti_usage        numeric(12,8) DEFAULT 1,
  ti_scenario     int REFERENCES scheduling.resource_timeline_scenario ( id ),
  ti_stat         scheduling.resource_timeline_blockstatus,

  CONSTRAINT dateend_smaller_datestart CHECK( ti_date_start <= ti_date_end )
);

-- Indizes
CREATE INDEX resource_timeline__a2_id__date_start ON scheduling.resource_timeline ( ti_a2_id, ti_date_start );
DROP INDEX IF EXISTS scheduling.resource_timeline__ti_resource_id;
--CREATE INDEX resource_timeline__ti_resource_id ON scheduling.resource_timeline ( ti_resource_id );
-- Die resource_timeline__ti_date_range__*-Indexe werden aktuell nicht wähend der Terminierung nicht verwemdet. Bei größeren
-- Datenmengen in der Tabelle resource_timeline__ti_date_range könnte der Planer sich aber für die Verwendung dieser Indexe
-- entscheiden, daher bleiben diese vorerst drin.
-- DROP INDEX IF EXISTS scheduling.resource_timeline__ti_resource_id__ti_date_range__btree;
CREATE INDEX resource_timeline__ti_resource_id__ti_date_range__btree ON scheduling.resource_timeline ( ti_resource_id, ti_date_start, ti_date_end DESC );
CREATE INDEX resource_timeline__ti_date_range__btree ON scheduling.resource_timeline ( ti_date_start, ti_date_end DESC );
CREATE INDEX resource_timeline__ti_date_range__spgist
    ON scheduling.resource_timeline USING SPGIST( tsrange( ti_date_start, ti_date_end, '[]' ) );
CREATE INDEX resource_timeline__ti_date_range__worktime
    ON scheduling.resource_timeline USING gist ( tsrange( ti_date_start::date, ti_date_end::date, '[]' ) );

CREATE INDEX resource_timeline__ti_date__termweek ON scheduling.resource_timeline ( termweek(ti_date_start) ); -- für Plantafelstatement
CREATE INDEX resource_timeline__date_end__type__resource ON scheduling.resource_timeline ( ti_date_end, ti_type, ti_resource_id );

-- Index resource_timeline__ti_type führt zu langsamen Terminierungen bei LOLL
DROP INDEX IF EXISTS resource_timeline__ti_type;
-- CREATE INDEX resource_timeline__ti_type ON scheduling.resource_timeline ( ti_type );
CREATE INDEX resource_timeline__a2_id__ti_type ON scheduling.resource_timeline ( ti_a2_id, ti_type );
CREATE INDEX resource_timeline__ti_type__ti_date_end__ti_a2_id ON scheduling.resource_timeline (ti_type, ti_date_end, ti_a2_id);

-- Trigger
-- Zurückschreiben der Terminierungszeitdaten in den AG bei automatischer Terminierung
DROP TRIGGER IF EXISTS resource_timeline__a_iu__termini ON scheduling.resource_timeline;
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__a_iu__termini() RETURNS TRIGGER AS $$
  DECLARE

      _prefix varchar := format( 'resource_timeline__a_iu__termini TI %L -', new.ti_id );
      _loglevel integer := TSystem.Log_Get_LogLevel( _user => 'yes' );

  BEGIN --ACHTUNG: nur wenn nicht aus manueller Terminierung kommend! siehe WHEN in Trigger => execution_flag__isset( _flagname => 'inTerminierung' )

      IF _loglevel >= 5 THEN
          RAISE NOTICE '% ENTER:%;', _prefix, clock_timestamp()::time;
      END IF;

      -- Nur für timeline-Einträge der Hauptressource.
      IF ( scheduling.resource_timeline__resource_id_main_resource__get( new.ti_a2_id ) = new.ti_resource_id ) THEN
          PERFORM scheduling.ab2__at_et__from__resource_timeline__sync( new.ti_a2_id );
      END IF;

      IF _loglevel >= 5 THEN
          RAISE NOTICE '% LEAVE:%;', _prefix, clock_timestamp()::time;
      END IF;

      RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER resource_timeline__a_iu__termini
    AFTER INSERT OR UPDATE
    OF ti_date_start, ti_date_end
    ON scheduling.resource_timeline
    FOR EACH ROW
    WHEN  (
                TSystem.execution_flag__isset__cascade_save( _flagname => 'inTerminierung' )  -- Nur während automatischer Terminierung. Verhinderung einer Triggerschleife zwischen "resource_timeline__a_iu__termini" und "ab2__a_iu__termination_manual".
            AND new.ti_type IN ( 'task', 'task.buffer' )                -- Nur für timeline-Einträge vom Typ 'task' oder 'task.buffer'.
          )
    EXECUTE PROCEDURE scheduling.resource_timeline__a_iu__termini();
--

-- Löschen der Terminierungsdaten aus dem AG bei automatischer (Aus-)Terminierung
DROP TRIGGER IF EXISTS resource_timeline__a_d__termini ON scheduling.resource_timeline;
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__a_d__termini() RETURNS TRIGGER AS $$
  DECLARE

      _prefix varchar := format( 'resource_timeline__a_d__termini TI %L -', old.ti_id );
      _loglevel integer := TSystem.Log_Get_LogLevel( _user => 'yes' );

  BEGIN --ACHTUNG: nur wenn nicht aus manueller Terminierung kommend! siehe WHEN in Trigger => execution_flag__isset( _flagname => 'inTerminierung' )

      IF _loglevel >= 5 THEN
          RAISE NOTICE '% ENTER:%;', _prefix, clock_timestamp()::time;
      END IF;

      -- Nur wenn keine timeline-Einträge mehr von der Hauptressource gefunden werden.
      IF ( scheduling.resource_timeline__resource_id_main_resource__get( old.ti_a2_id ) IS null ) THEN
          -- Spalten Zeiten in AG NULLen.
          UPDATE ab2
             SET a2_at = null,
                 a2_et = null -- setzt automatisch a2_interm
           WHERE a2_id = old.ti_a2_id
             AND NOT a2_ende;
      END IF;

      IF _loglevel >= 5 THEN
          RAISE NOTICE '% LEAVE:%;', _prefix, clock_timestamp()::time;
      END IF;

      RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER resource_timeline__a_d__termini
    AFTER DELETE
    ON scheduling.resource_timeline
    FOR EACH ROW
    WHEN  (
                TSystem.execution_flag__isset__cascade_save( _flagname => 'inTerminierung' )  -- Nur während automatischer Terminierung. Verhinderung einer Triggerschleife zwischen "resource_timeline__a_iu__termini" und "ab2__a_iu__termination_manual".
            AND old.ti_type IN ( 'task', 'task.buffer' )                -- Nur für timeline-Einträge vom Typ 'task' oder 'task.buffer'.
          )
    EXECUTE PROCEDURE scheduling.resource_timeline__a_d__termini();
--

CREATE TABLE scheduling.resource_termination_results (

    id                       serial PRIMARY KEY,
    search_id                int NOT null,
    abk_ab_ix                int NOT null references abk( ab_ix ) ON DELETE CASCADE,

    -- paths for searching the results
    resource_id_path         int[] NOT null,
    resource_vertices_path  int[] NOT null,
    ab2_a2_id_path           int[] NOT null,

    -- start and finish time of abk termination
    date_start               timestamp NOT null,
    date_end                 timestamp NOT null,

    -- jsonb encoded ab2_termination_resultset which must be unnested to get the actual timeslots for scheduling.resource_timeline
    ab2_termination_result jsonb[]
);

CREATE INDEX resource_termination_results_search_id_idx ON scheduling.resource_termination_results ( search_id );
CREATE INDEX resource_termination_results_ab2_a2_id_path_ab2_a2_id_path1_idx ON scheduling.resource_termination_results (
    ( ab2_a2_id_path[ array_length( ab2_a2_id_path , 1) ] ),
    ( ab2_a2_id_path[ 1 ] ),
    search_id
);

CREATE SEQUENCE scheduling.resource_termination_sequence;

-- transportmatrix
CREATE TABLE scheduling.resource_transport (
  id                    serial PRIMARY KEY,

  -- which resource is transported
  resource_id           int REFERENCES scheduling.resource( id ) ON DELETE CASCADE,

  -- from one resource
  resource_id_from      int NOT null REFERENCES scheduling.resource( id ) ON DELETE CASCADE,

  -- to another resource
  resource_id_to        int NOT null  REFERENCES scheduling.resource( id ) ON DELETE CASCADE,

  -- transported by
  resource_id_via       int REFERENCES scheduling.resource( id ) ON DELETE CASCADE,

  -- required worktime for transport
  work_required         int NOT null DEFAULT 0,

  -- userinput in days. see trigger
  buffer_time_user_hours numeric,

  buffer_time           int NOT null DEFAULT 0,
  buffer_time_default   int,

  -- marks entry as a default generated element which is used to distinguish
  -- custom elements in a performant manner during updates
  default_rule          bool NOT null DEFAULT true,

  -- enforce generic uniqueness
  EXCLUDE (
      resource_id_from WITH =,
      resource_id_to WITH =
  ) WHERE (
        resource_id IS null
    AND resource_id_via IS null
  ),

  -- enforce resource specific uniqueness, without transportation
  EXCLUDE (
      resource_id_from WITH =,
      resource_id_to WITH = ,
      resource_id WITH =
  ) WHERE (
      resource_id_via IS null
  ),

  -- enforce generic uniqueness, with transportation
  EXCLUDE (
      resource_id_from WITH =,
      resource_id_to WITH = ,
      resource_id_via WITH =
  ) WHERE (
      resource_id IS null
  ),

  -- enforce work_time only be present , when a transporting resource is present
  CHECK (
        work_required = 0
    OR (
             resource_id_via IS NOT null
         AND work_required > 0
       )
  )
);

-- this will not check for NULLS
CREATE UNIQUE INDEX resource_transport_resource_id_resource_id_from_resource_id_idx ON scheduling.resource_transport ( resource_id, resource_id_from, resource_id_to, resource_id_via );
CREATE INDEX resource_transport_resource_id_from_resource_id_to_idx ON scheduling.resource_transport ( resource_id_from, resource_id_to );
CREATE INDEX resource_transport_default_rule_resource_id_from_idx ON scheduling.resource_transport ( default_rule, resource_id_from );
CREATE INDEX resource_transport_default_rule_resource_id_to_idx ON scheduling.resource_transport ( default_rule, resource_id_to );


CREATE OR REPLACE FUNCTION scheduling.ksvba__b_u__buffer_time_user_hours() RETURNS TRIGGER AS $$
  BEGIN
    IF new.buffer_time_user_hours IS NOT null THEN
        new.buffer_time := new.buffer_time_user_hours * 60 * 60; -- hour, day
    ELSE
        new.buffer_time := new.buffer_time_default;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ksvba__b_u__buffer_time_user_hours
    BEFORE UPDATE OF buffer_time_user_hours
    ON scheduling.resource_transport
    FOR EACH ROW
    EXECUTE PROCEDURE scheduling.ksvba__b_u__buffer_time_user_hours();


CREATE OR REPLACE FUNCTION scheduling.ksvba__a_50_iu__resource_transport_sync() RETURNS trigger AS $$
  DECLARE _resource_id  int    := id FROM scheduling.resource WHERE context = 'ksvba' AND context_id = new.ksb_id;
  BEGIN

      IF ks_sperr FROM ksv WHERE ks_id = new.ksb_ks_id THEN
        DELETE FROM scheduling.resource_transport
              WHERE   resource_id_from = _resource_id
                   OR resource_id_to = _resource_id
        ;
        RETURN new;
      END IF;

      IF ( TG_OP = 'INSERT' ) THEN
          PERFORM scheduling.ksvba__resource__transport_update( new, null );
      END IF;

      IF ( TG_OP = 'UPDATE' ) THEN
          PERFORM scheduling.ksvba__resource__transport_update( new, old );
      END IF;

      RETURN new;

  END $$ LANGUAGE plpgsql;

-- trigger for syncing matrix data
-- delete is covered via FK
CREATE TRIGGER ksvba__a_50_iu__resource_transport_sync
  AFTER UPDATE OR INSERT
  ON ksvba
  FOR EACH ROW
  EXECUTE PROCEDURE scheduling.ksvba__a_50_iu__resource_transport_sync();





CREATE TABLE scheduling.scheduled_results_storage__abk
  (
    srs_id                  serial PRIMARY KEY,
    srs_ab_ix               integer,
    srs_ab_at               date,
    srs_ab_et               date,
    srs_user                varchar(100),
    srs_timestamp           timestamp(0)
  );


CREATE OR REPLACE FUNCTION scheduling.scheduled_results_storage__abk__create() RETURNS VOID
  AS $$
  BEGIN
    DELETE FROM scheduling.scheduled_results_storage__abk;
    INSERT INTO scheduling.scheduled_results_storage__abk
                (srs_ab_ix, srs_ab_at, srs_ab_et)
         SELECT ab_ix, ab_at, ab_et FROM abk WHERE NOT ab_done AND ab_at IS NOT null;
  END $$ LANGUAGE plpgsql;

--- #20330
CREATE TABLE scheduling.scheduling_log (
  sl_id                     serial       PRIMARY KEY,
  sl_time                   timestamp(0) NOT NULL DEFAULT now(),
  sl_tablename              varchar(100) NOT NULL,
  sl_dbrid                  varchar(32),
  sl_json                   jsonb
  );

CREATE UNIQUE INDEX scheduling_log_unique ON scheduling.scheduling_log ( sl_time, sl_tablename, sl_dbrid );
